-- قاعدة بيانات موقع تأجير السيارات CarBook
-- إنشاء قاعدة البيانات
CREATE DATABASE IF NOT EXISTS carbook_db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE carbook_db;

-- جدول المستخدمين
CREATE TABLE users (
    user_id INT PRIMARY KEY AUTO_INCREMENT,
    full_name VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    phone VARCHAR(20),
    password_hash VARCHAR(255) NOT NULL,
    address TEXT,
    city VARCHAR(50),
    country VARCHAR(50),
    user_type ENUM('customer', 'driver', 'admin') DEFAULT 'customer',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    is_active BOOLEAN DEFAULT TRUE,
    INDEX idx_email (email),
    INDEX idx_user_type (user_type)
) ENGINE=InnoDB;

-- جدول فئات السيارات
CREATE TABLE car_categories (
    category_id INT PRIMARY KEY AUTO_INCREMENT,
    category_name VARCHAR(50) NOT NULL,
    description TEXT,
    icon_class VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- جدول السيارات
CREATE TABLE cars (
    car_id INT PRIMARY KEY AUTO_INCREMENT,
    car_name VARCHAR(100) NOT NULL,
    brand VARCHAR(50) NOT NULL,
    model VARCHAR(50) NOT NULL,
    year INT,
    category_id INT,
    license_plate VARCHAR(20) UNIQUE,
    color VARCHAR(30),
    seats INT DEFAULT 5,
    transmission ENUM('Manual', 'Automatic') DEFAULT 'Manual',
    fuel_type ENUM('Petrol', 'Diesel', 'Electric', 'Hybrid') DEFAULT 'Petrol',
    mileage INT DEFAULT 0,
    luggage_capacity INT DEFAULT 4,
    image_url VARCHAR(255),
    hourly_rate DECIMAL(10, 2),
    daily_rate DECIMAL(10, 2) NOT NULL,
    monthly_rate DECIMAL(10, 2),
    fuel_surcharge DECIMAL(10, 2) DEFAULT 3.00,
    is_available BOOLEAN DEFAULT TRUE,
    status ENUM('available', 'rented', 'maintenance', 'retired') DEFAULT 'available',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES car_categories(category_id),
    INDEX idx_brand (brand),
    INDEX idx_status (status),
    INDEX idx_availability (is_available)
) ENGINE=InnoDB;

-- جدول ميزات السيارات
CREATE TABLE car_features (
    feature_id INT PRIMARY KEY AUTO_INCREMENT,
    car_id INT NOT NULL,
    feature_name VARCHAR(50) NOT NULL,
    is_available BOOLEAN DEFAULT TRUE,
    FOREIGN KEY (car_id) REFERENCES cars(car_id) ON DELETE CASCADE,
    INDEX idx_car_id (car_id)
) ENGINE=InnoDB;

-- جدول الحجوزات
CREATE TABLE bookings (
    booking_id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    car_id INT NOT NULL,
    pickup_location VARCHAR(255) NOT NULL,
    dropoff_location VARCHAR(255) NOT NULL,
    pickup_date DATE NOT NULL,
    pickup_time TIME NOT NULL,
    dropoff_date DATE NOT NULL,
    dropoff_time TIME,
    rental_type ENUM('hourly', 'daily', 'monthly') DEFAULT 'daily',
    total_amount DECIMAL(10, 2) NOT NULL,
    discount_amount DECIMAL(10, 2) DEFAULT 0,
    final_amount DECIMAL(10, 2) NOT NULL,
    status ENUM('pending', 'confirmed', 'active', 'completed', 'cancelled') DEFAULT 'pending',
    payment_status ENUM('pending', 'paid', 'refunded') DEFAULT 'pending',
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(user_id),
    FOREIGN KEY (car_id) REFERENCES cars(car_id),
    INDEX idx_user_id (user_id),
    INDEX idx_car_id (car_id),
    INDEX idx_status (status),
    INDEX idx_pickup_date (pickup_date)
) ENGINE=InnoDB;

-- جدول المدفوعات
CREATE TABLE payments (
    payment_id INT PRIMARY KEY AUTO_INCREMENT,
    booking_id INT NOT NULL,
    amount DECIMAL(10, 2) NOT NULL,
    payment_method ENUM('cash', 'credit_card', 'debit_card', 'bank_transfer', 'online') NOT NULL,
    payment_status ENUM('pending', 'completed', 'failed', 'refunded') DEFAULT 'pending',
    transaction_id VARCHAR(100),
    payment_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    notes TEXT,
    FOREIGN KEY (booking_id) REFERENCES bookings(booking_id),
    INDEX idx_booking_id (booking_id),
    INDEX idx_payment_status (payment_status)
) ENGINE=InnoDB;

-- جدول التقييمات والمراجعات
CREATE TABLE reviews (
    review_id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    car_id INT NOT NULL,
    booking_id INT,
    rating INT CHECK (rating >= 1 AND rating <= 5),
    review_text TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    is_approved BOOLEAN DEFAULT FALSE,
    FOREIGN KEY (user_id) REFERENCES users(user_id),
    FOREIGN KEY (car_id) REFERENCES cars(car_id),
    FOREIGN KEY (booking_id) REFERENCES bookings(booking_id),
    INDEX idx_car_id (car_id),
    INDEX idx_rating (rating)
) ENGINE=InnoDB;

-- جدول الخدمات
CREATE TABLE services (
    service_id INT PRIMARY KEY AUTO_INCREMENT,
    service_name VARCHAR(100) NOT NULL,
    description TEXT,
    icon_class VARCHAR(50),
    image_url VARCHAR(255),
    is_active BOOLEAN DEFAULT TRUE,
    display_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- جدول المقالات/المدونة
CREATE TABLE blog_posts (
    post_id INT PRIMARY KEY AUTO_INCREMENT,
    title VARCHAR(200) NOT NULL,
    slug VARCHAR(200) UNIQUE NOT NULL,
    content TEXT NOT NULL,
    excerpt TEXT,
    featured_image VARCHAR(255),
    author_id INT,
    category VARCHAR(50),
    tags VARCHAR(255),
    views_count INT DEFAULT 0,
    status ENUM('draft', 'published', 'archived') DEFAULT 'draft',
    published_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (author_id) REFERENCES users(user_id),
    INDEX idx_slug (slug),
    INDEX idx_status (status)
) ENGINE=InnoDB;

-- جدول التواصل/الاستفسارات
CREATE TABLE contact_messages (
    message_id INT PRIMARY KEY AUTO_INCREMENT,
    full_name VARCHAR(100) NOT NULL,
    email VARCHAR(100) NOT NULL,
    phone VARCHAR(20),
    subject VARCHAR(200),
    message TEXT NOT NULL,
    status ENUM('new', 'read', 'replied', 'closed') DEFAULT 'new',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_status (status),
    INDEX idx_email (email)
) ENGINE=InnoDB;

-- جدول الإشعارات
CREATE TABLE notifications (
    notification_id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    title VARCHAR(100) NOT NULL,
    message TEXT NOT NULL,
    type ENUM('booking', 'payment', 'system', 'promotion') DEFAULT 'system',
    is_read BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(user_id),
    INDEX idx_user_id (user_id),
    INDEX idx_is_read (is_read)
) ENGINE=InnoDB;

-- جدول إحصائيات الموقع
CREATE TABLE site_statistics (
    stat_id INT PRIMARY KEY AUTO_INCREMENT,
    years_experience INT DEFAULT 60,
    total_cars INT DEFAULT 1090,
    happy_customers INT DEFAULT 2590,
    total_branches INT DEFAULT 67,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- إدراج بيانات أولية للفئات
INSERT INTO car_categories (category_name, description, icon_class) VALUES
('Chevrolet', 'سيارات شيفروليه الفاخرة', 'flaticon-car'),
('Subaru', 'سيارات سوبارو العائلية', 'flaticon-transportation'),
('SUV', 'سيارات الدفع الرباعي', 'flaticon-wedding-car');

-- إدراج بيانات أولية للسيارات
INSERT INTO cars (car_name, brand, model, year, category_id, seats, transmission, fuel_type, mileage, luggage_capacity, image_url, hourly_rate, daily_rate, monthly_rate) VALUES
('Mercedes Grand Sedan', 'Chevrolet', 'Grand Sedan', 2023, 1, 5, 'Manual', 'Petrol', 40000, 4, 'images/car-1.jpg', 10.99, 60.99, 995.99),
('Range Rover', 'Subaru', 'Range Rover', 2023, 2, 5, 'Automatic', 'Diesel', 35000, 4, 'images/car-2.jpg', 10.99, 60.99, 995.99),
('Mercedes Grand Sedan', 'Chevrolet', 'Grand Sedan', 2022, 1, 5, 'Manual', 'Petrol', 42000, 4, 'images/car-3.jpg', 10.99, 60.99, 995.99);

-- إدراج بيانات أولية للميزات
INSERT INTO car_features (car_id, feature_name, is_available) VALUES
(1, 'Airconditions', TRUE),
(1, 'Child Seat', TRUE),
(1, 'GPS', TRUE),
(1, 'Luggage', TRUE),
(1, 'Music', TRUE),
(1, 'Seat Belt', TRUE),
(1, 'Sleeping Bed', FALSE),
(1, 'Water', TRUE),
(1, 'Bluetooth', TRUE),
(1, 'Onboard computer', FALSE);

-- إدراج بيانات أولية للخدمات
INSERT INTO services (service_name, description, icon_class) VALUES
('Wedding Ceremony', 'خدمة تأجير السيارات لحفلات الزفاف', 'flaticon-wedding-car'),
('City Transfer', 'خدمة التنقل داخل المدينة', 'flaticon-transportation'),
('Airport Transfer', 'خدمة التوصيل من وإلى المطار', 'flaticon-car'),
('Whole City Tour', 'جولات سياحية شاملة في المدينة', 'flaticon-transportation');

-- إدراج بيانات الإحصائيات
INSERT INTO site_statistics (years_experience, total_cars, happy_customers, total_branches) VALUES
(60, 1090, 2590, 67);

-- إنشاء مستخدم مسؤول افتراضي (كلمة المرور: admin123 - يجب تغييرها)
INSERT INTO users (full_name, email, phone, password_hash, user_type, city, country) VALUES
('Admin User', 'admin@carbook.com', '+1234567890', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin', 'San Francisco', 'USA');

-- Views لسهولة الاستعلام
-- عرض السيارات المتاحة مع تفاصيلها
CREATE VIEW available_cars_view AS
SELECT 
    c.car_id,
    c.car_name,
    c.brand,
    c.model,
    c.year,
    cat.category_name,
    c.seats,
    c.transmission,
    c.fuel_type,
    c.image_url,
    c.daily_rate,
    c.hourly_rate,
    c.monthly_rate,
    c.fuel_surcharge,
    AVG(r.rating) as avg_rating,
    COUNT(r.review_id) as total_reviews
FROM cars c
LEFT JOIN car_categories cat ON c.category_id = cat.category_id
LEFT JOIN reviews r ON c.car_id = r.car_id
WHERE c.is_available = TRUE AND c.status = 'available'
GROUP BY c.car_id;

-- عرض الحجوزات النشطة
CREATE VIEW active_bookings_view AS
SELECT 
    b.booking_id,
    b.pickup_date,
    b.dropoff_date,
    b.status,
    u.full_name as customer_name,
    u.email as customer_email,
    u.phone as customer_phone,
    c.car_name,
    c.brand,
    c.license_plate,
    b.total_amount,
    b.payment_status
FROM bookings b
JOIN users u ON b.user_id = u.user_id
JOIN cars c ON b.car_id = c.car_id
WHERE b.status IN ('confirmed', 'active');

-- Stored Procedures
-- إجراء لحساب تكلفة الحجز
DELIMITER //
CREATE PROCEDURE calculate_booking_cost(
    IN p_car_id INT,
    IN p_rental_type VARCHAR(20),
    IN p_pickup_date DATE,
    IN p_dropoff_date DATE,
    OUT p_total_amount DECIMAL(10,2)
)
BEGIN
    DECLARE v_daily_rate DECIMAL(10,2);
    DECLARE v_hourly_rate DECIMAL(10,2);
    DECLARE v_monthly_rate DECIMAL(10,2);
    DECLARE v_days INT;
    
    SELECT daily_rate, hourly_rate, monthly_rate 
    INTO v_daily_rate, v_hourly_rate, v_monthly_rate
    FROM cars 
    WHERE car_id = p_car_id;
    
    SET v_days = DATEDIFF(p_dropoff_date, p_pickup_date);
    
    IF p_rental_type = 'daily' THEN
        SET p_total_amount = v_daily_rate * v_days;
    ELSEIF p_rental_type = 'monthly' THEN
        SET p_total_amount = v_monthly_rate * CEIL(v_days / 30);
    ELSE
        SET p_total_amount = v_hourly_rate * v_days * 24;
    END IF;
END //

-- إجراء لتحديث حالة السيارة
CREATE PROCEDURE update_car_status(
    IN p_car_id INT,
    IN p_status VARCHAR(20)
)
BEGIN
    UPDATE cars 
    SET status = p_status,
        is_available = CASE 
            WHEN p_status = 'available' THEN TRUE 
            ELSE FALSE 
        END
    WHERE car_id = p_car_id;
END //

DELIMITER ;

-- Triggers
-- تحديث حالة السيارة تلقائياً عند إنشاء حجز
DELIMITER //
CREATE TRIGGER after_booking_insert
AFTER INSERT ON bookings
FOR EACH ROW
BEGIN
    IF NEW.status = 'confirmed' THEN
        UPDATE cars 
        SET status = 'rented', is_available = FALSE 
        WHERE car_id = NEW.car_id;
    END IF;
END //

-- تحديث حالة السيارة عند إكمال الحجز
CREATE TRIGGER after_booking_complete
AFTER UPDATE ON bookings
FOR EACH ROW
BEGIN
    IF NEW.status = 'completed' AND OLD.status != 'completed' THEN
        UPDATE cars 
        SET status = 'available', is_available = TRUE 
        WHERE car_id = NEW.car_id;
    END IF;
END //

DELIMITER ;

-- Indexes للأداء
CREATE INDEX idx_booking_dates ON bookings(pickup_date, dropoff_date);
CREATE INDEX idx_car_rates ON cars(daily_rate, hourly_rate, monthly_rate);
CREATE INDEX idx_review_approved ON reviews(is_approved, created_at);

-- Full-text search للبحث
ALTER TABLE cars ADD FULLTEXT INDEX ft_car_search (car_name, brand, model);
ALTER TABLE blog_posts ADD FULLTEXT INDEX ft_blog_search (title, content);